# vim: set ts=8 sw=4 sts=4 et ai:
"""
Add natural keys to contrib.auth.User and Group models.
https://code.djangoproject.com/ticket/13914

In this case, only to the Group model we won't create duplicate groups
without having to hardcode the Group PKs.

In project.fixtures.initial_data.xml we define which groups have which
permissions. But the extension modules will require additional groups
to be created. They cannot safely choose a PK without stomping on
different extension PKs.

The fix: remove the PK from the fixtures.

Ticket #13914 has been closed, the fix has been committed as
954e3b4ad37b572cdc46baf3e35c712f4049efea in branch 1.7b4.
"""
from django import VERSION

if VERSION < (1, 7):
    from django.contrib.auth import models as auth_models
    from django.db import models as db_models

    class GroupManagerWithNaturalKey(db_models.Manager):
        def get_by_natural_key(self, name):
            return self.get(name=name)
    #auth_models.Group.objects = GroupManagerWithNaturalKey()
    auth_models.Group._default_manager = GroupManagerWithNaturalKey()
    auth_models.Group._default_manager.model = auth_models.Group

    def group_natural_key(self):
        return (self.name,)
    auth_models.Group.natural_key = group_natural_key

    # But wait, there is more. We also need to patch the xml
    # Deserializer to accept the natural key as PK.
    from django.core.serializers import base, xml_serializer
    from django.db import models

    # This function is stolen from Django 1.7.
    def build_instance(Model, data, db):
        """
        Build a model instance.

        If the model instance doesn't have a primary key and the model supports
        natural keys, try to retrieve it from the database.
        """
        obj = Model(**data)
        if (obj.pk is None and hasattr(Model, 'natural_key') and
                hasattr(Model._default_manager, 'get_by_natural_key')):
            natural_key = obj.natural_key()
            try:
                obj.pk = Model._default_manager.db_manager(db).get_by_natural_key(*natural_key).pk
            except Model.DoesNotExist:
                pass
        return obj

    # This function is unchanged from
    # django.core.serializers.xml_serializer.Deserializer
    # apart from the build_instance code at the bottom.
    def deserializer_handle_object(self, node):
        """
        Convert an <object> node to a DeserializedObject.
        """
        # Look up the model using the model loading mechanism. If this fails,
        # bail.
        Model = self._get_model_from_node(node, "model")

        # Start building a data dictionary from the object.
        # If the node is missing the pk set it to None
        if node.hasAttribute("pk"):
            pk = node.getAttribute("pk")
        else:
            pk = None

        data = {Model._meta.pk.attname: Model._meta.pk.to_python(pk)}

        # Also start building a dict of m2m data (this is saved as
        # {m2m_accessor_attribute : [list_of_related_objects]})
        m2m_data = {}

        # Deseralize each field.
        for field_node in node.getElementsByTagName("field"):
            # If the field is missing the name attribute, bail (are you
            # sensing a pattern here?)
            field_name = field_node.getAttribute("name")
            if not field_name:
                raise base.DeserializationError("<field> node is missing the 'name' attribute")

            # Get the field from the Model. This will raise a
            # FieldDoesNotExist if, well, the field doesn't exist, which will
            # be propagated correctly.
            field = Model._meta.get_field(field_name)

            # As is usually the case, relation fields get the special treatment.
            if field.rel and isinstance(field.rel, models.ManyToManyRel):
                m2m_data[field.name] = self._handle_m2m_field_node(field_node, field)
            elif field.rel and isinstance(field.rel, models.ManyToOneRel):
                data[field.attname] = self._handle_fk_field_node(field_node, field)
            else:
                if field_node.getElementsByTagName('None'):
                    value = None
                else:
                    value = field.to_python(xml_serializer.getInnerText(field_node).strip())
                data[field.name] = value

        # Original.
        # # Return a DeserializedObject so that the m2m data has a place to live.
        # return base.DeserializedObject(Model(**data), m2m_data)

        # New.
        obj = build_instance(Model, data, self.db)
        # Return a DeserializedObject so that the m2m data has a place to live.
        return base.DeserializedObject(obj, m2m_data)

    # Overwrite the old method with the new.
    xml_serializer.Deserializer._handle_object = deserializer_handle_object
