"""
Firebird database backend for Django.

Requires kinterbasdb: http://kinterbasdb.sourceforge.net/
"""

from django.conf import settings
from django.db.backends import util
import re
try:
    import kinterbasdb as Database
except ImportError, e:
    from django.core.exceptions import ImproperlyConfigured
    raise ImproperlyConfigured, "Error loading kinterbasdb module: %s" % e

DatabaseError = Database.DatabaseError
Database.init(type_conv=200)

try:
    # Only exists in Python 2.4+
    from threading import local
except ImportError:
    # Import copy of _thread_local.py from Python 2.4
    from django.utils._threading_local import local

class DatabaseWrapper(local):
    def __init__(self, **kwargs):
        self.connection = None
        self.queries = []
        self.options = kwargs

    def _valid_connection(self):
        return self.connection is not None

    def cursor(self):
        if not self._valid_connection():
            if settings.DATABASE_NAME == '':
                from django.core.exceptions import ImproperlyConfigured
                raise ImproperlyConfigured, "You need to specify DATABASE_NAME in your Django settings file."

            self.connection = Database.connect(
                database=settings.DATABASE_NAME,
                user=settings.DATABASE_USER,
                password=settings.DATABASE_PASSWORD,
                host=settings.DATABASE_HOST
            )
        cursor = FormatPlaceholderStylerCursor(self.connection)
        if settings.DEBUG:
            return util.CursorDebugWrapper(cursor, self)
        return cursor

    def _commit(self):
        return self.connection.commit()

    def _rollback(self):
        if self.connection:
            return self.connection.rollback()

    def close(self):
        if self.connection is not None:
            self.connection.close()
            self.connection = None

    def _drop_database(self):
        return self.connection.drop_database()

allows_group_by_ordinal = True
allows_unique_and_pk = True        # Suppress UNIQUE/PK for Oracle (ORA-02259)
needs_datetime_string_cast = False
supports_constraints = True
uses_case_insensitive_names = True

class FormatPlaceholderStylerCursor(Database.Cursor):
    """
    Django uses "format" ('%s') style placeholders, but firebird uses "qmark" ('?') style.
    This fixes it -- but note that if you want to use a literal "%s" in a query,
    you'll need to use "%%s".
    """
    def execute(self, query, params=[]):
        #print "%s - %s" % (query, params)
        return Database.Cursor.execute(self, self._convert_placeholders(query, len(params)), params)

    def executemany(self, query, params=[]):
        return Database.Cursor.executemany(self, self._convert_placeholders(query, len(params[0])), params)

    def _convert_placeholders(self, query, num_params):
        return query % tuple("?" * num_params)

#try:
#    _quote_name = settings.DATABASE_QUOTE_NAMES
#except AttributeError:
#    _quote_name = True

def quote_name(name):
    # the standard for firebird is not to quote names but in django
    # it will quote all names uppercased so we can write sql without quotes,
    # like oracle truncate names bigger than 30 chars
    name = name.upper()
    if not name.startswith('"') and not name.endswith('"'):
        name = '"%s"' % util.truncate_name(name.upper(), get_max_name_length())
    return name

#dictfetchone = util.dictfetchone
#dictfetchmany = util.dictfetchmany
#dictfetchall  = util.dictfetchall

def dictfetchone(cursor):
    "Returns a row from the cursor as a dict"
    return cursor.dictfetchone()

def dictfetchmany(cursor, number):
    "Returns a certain number of rows from a cursor as a dict"
    return cursor.dictfetchmany(number)

def dictfetchall(cursor):
    "Returns all rows from a cursor as a dict"
    return cursor.dictfetchall()

def get_last_insert_id(cursor, table_name, pk_name):
    generator_name = "%s_gn" % util.truncate_name(table_name, get_max_name_length()-3)
    cursor.execute('SELECT GEN_ID(%s, 0) FROM %s' % (quote_name(generator_name), quote_name(table_name)))
    return cursor.fetchone()[0]

def get_date_extract_sql(lookup_type, table_name):
    # lookup_type is 'year', 'month', 'day'
    return "EXTRACT(%s FROM %s)" % (lookup_type.upper(), table_name)

def get_date_trunc_sql(lookup_type, field_name):
    if lookup_type == 'year':
         sql = "EXTRACT(year FROM %s)||'-01-01 00:00:00'" % field_name
    elif lookup_type == 'month':
        sql = "EXTRACT(year FROM %s)||'-'||EXTRACT(month FROM %s)||'-01 00:00:00'" % (field_name, field_name)
    elif lookup_type == 'day':
        sql = "EXTRACT(year FROM %s)||'-'||EXTRACT(month FROM %s)||'-'||EXTRACT(day FROM %s)||' 00:00:00'" % (field_name, field_name, field_name)
    return "CAST(%s AS TIMESTAMP)" % sql

def get_datetime_cast_sql():
    return None

def get_limit_offset_sql(limit, offset=None):
    # Limits and offset are too complicated to be handled here.
    # Instead, they are handled in django/db/backends/firebird/query.py.
    raise NotImplementedError

def get_random_function_sql():
    # First must install ib_udf (comes with the server)
    # DECLARE EXTERNAL FUNCTION rand
    #   RETURNS DOUBLE PRECISION
    #   BY VALUE ENTRY_POINT 'IB_UDF_rand' MODULE_NAME 'ib_udf';
    return "RAND()"

def get_fulltext_search_sql(field_name):
    raise NotImplementedError

def get_drop_foreignkey_sql():
    return "DROP CONSTRAINT"

def get_pk_default_value():
    return "NULL"

def get_max_name_length():
    return 30

def get_autoinc_sql(table_name):
    # To simulate auto-incrementing primary keys in Firebird, we have to
    # create a generator (sequence in Firebird 2) and a trigger.
    field_name = 'ID'
    generator_name = "%s_GN" % util.truncate_name(table_name, get_max_name_length()-3).upper()
    trigger_name = "%s_TR" % util.truncate_name(table_name, get_max_name_length()-3).upper()
    generator_sql = 'CREATE GENERATOR %s;' % quote_name(generator_name)
    trigger_sql = []
    trigger_sql.append("CREATE TRIGGER %s FOR %s" % (trigger_name, table_name))
    trigger_sql.append("ACTIVE BEFORE INSERT POSITION 0")
    trigger_sql.append("AS")
    trigger_sql.append("BEGIN")
    trigger_sql.append("  if (NEW.%s IS NULL) THEN" % field_name)
    trigger_sql.append("  BEGIN")
    trigger_sql.append("    NEW.%s = GEN_ID(%s, 1);" % (field_name, generator_name))
    trigger_sql.append("  END")
    trigger_sql.append("END")
    return generator_sql, "\n".join(trigger_sql)

OPERATOR_MAPPING = {
    'exact': '= %s',
    'iexact': 'LIKE UPPER(%s)', # CONTAINIG
    'contains': 'LIKE %s',
    'icontains': 'LIKE UPPER(%s)', # CONTAINIG
    'gt': '> %s',
    'gte': '>= %s',
    'lt': '< %s',
    'lte': '<= %s',
    'startswith': 'LIKE %s', # STARTING WITH
    'endswith': 'LIKE %s',
    'istartswith': 'LIKE UPPER(%s)',
    'iendswith': 'LIKE UPPER(%s)',
}
