from itertools import izip
from django.db import models
from django.db.backends.util import truncate_name
from django.db.backends.oracle import compiler
from django.db.models.sql.constants import TABLE_NAME
from django.db.models.sql.query import get_proxied_model

SQLCompiler = compiler.SQLCompiler

class CustomSQLCompiler(compiler.SQLCompiler):
    """CustomSQLCompiler to demonstrate problem with select_related() method"""
    
    def get_columns(self, with_aliases=False):
        """Custom get_columns. Hacked to mangle SQL so that all models.CharField instances are
        forced to uppercase by wrapping value in SQL upper() function"""
        
        qn = self.quote_name_unless_alias
        qn2 = self.connection.ops.quote_name
        result = ['(%s) AS %s' % (col[0], qn2(alias)) for alias, col in self.query.extra_select.iteritems()]
        aliases = set(self.query.extra_select.keys())
        if with_aliases:
            col_aliases = aliases.copy()
        else:
            col_aliases = set()
        if self.query.select:
            only_load = self.deferred_to_columns()
            for col, field in izip(self.query.select, self.query.select_fields):
                if isinstance(col, (list, tuple)):
                    alias, column = col
                    table = self.query.alias_map[alias][TABLE_NAME]
                    if table in only_load and col not in only_load[table]:
                        continue
                    
                    # Mangle query for models.CharField to demonstrate #12344
                    # For other field types keep them as they are.
                    if isinstance(field, models.CharField):
                        r = 'upper(%s.%s)' % (qn(alias), qn(column))
                    else:
                        r = '%s.%s' % (qn(alias), qn(column))
                        
                    if with_aliases:
                        if col[1] in col_aliases:
                            c_alias = 'Col%d' % len(col_aliases)
                            result.append('%s AS %s' % (r, c_alias))
                            aliases.add(c_alias)
                            col_aliases.add(c_alias)
                        else:
                            result.append('%s AS %s' % (r, qn2(col[1])))
                            aliases.add(r)
                            col_aliases.add(col[1])
                    else:
                        result.append(r)
                        aliases.add(r)
                        col_aliases.add(col[1])
                else:
                    result.append(col.as_sql(qn, self.connection))

                    if hasattr(col, 'alias'):
                        aliases.add(col.alias)
                        col_aliases.add(col.alias)

        elif self.query.default_cols:
            cols, new_aliases = self.get_default_columns(with_aliases,
                    col_aliases)
            result.extend(cols)
            aliases.update(new_aliases)

        max_name_length = self.connection.ops.max_name_length()
        result.extend([
            '%s%s' % (
                aggregate.as_sql(qn, self.connection),
                alias is not None
                    and ' AS %s' % qn(truncate_name(alias, max_name_length))
                    or ''
            )
            for alias, aggregate in self.query.aggregate_select.items()
        ])

        # We do not mangle for related cols here, it's not relevant since
        # this code is actually never called in test case. 
        for table, col in self.query.related_select_cols:
            raise Exception('Should not be here')
            r = '%s.%s' % (qn(table), qn(col))
            if with_aliases and col in col_aliases:
                c_alias = 'Col%d' % len(col_aliases)
                result.append('%s AS %s' % (r, c_alias))
                aliases.add(c_alias)
                col_aliases.add(c_alias)
            else:
                result.append(r)
                aliases.add(r)
                col_aliases.add(col)

        self._select_aliases = aliases
        return result

    def get_default_columns(self, with_aliases=False, col_aliases=None,
            start_alias=None, opts=None, as_pairs=False):
        """
        Computes the default columns for selecting every field in the base
        model. Will sometimes be called to pull in related models (e.g. via
        select_related), in which case "opts" and "start_alias" will be given
        to provide a starting point for the traversal.

        Returns a list of strings, quoted appropriately for use in SQL
        directly, as well as a set of aliases used in the select statement (if
        'as_pairs' is True, returns a list of (alias, col_name) pairs instead
        of strings as the first component and None as the second component).
        """
        result = []
        if opts is None:
            opts = self.query.model._meta
        qn = self.quote_name_unless_alias
        qn2 = self.connection.ops.quote_name
        aliases = set()
        only_load = self.deferred_to_columns()
        # Skip all proxy to the root proxied model
        proxied_model = get_proxied_model(opts)

        if start_alias:
            seen = {None: start_alias}
        for field, model in opts.get_fields_with_model():
            if start_alias:
                try:
                    alias = seen[model]
                except KeyError:
                    if model is proxied_model:
                        alias = start_alias
                    else:
                        link_field = opts.get_ancestor_link(model)
                        alias = self.query.join((start_alias, model._meta.db_table,
                                link_field.column, model._meta.pk.column))
                    seen[model] = alias
            else:
                # If we're starting from the base model of the queryset, the
                # aliases will have already been set up in pre_sql_setup(), so
                # we can save time here.
                alias = self.query.included_inherited_models[model]
            table = self.query.alias_map[alias][TABLE_NAME]
            if table in only_load and field.column not in only_load[table]:
                continue
            if as_pairs:
                result.append((alias, field.column))
                aliases.add(alias)
                continue
            
            # Customized to demonstrate #12344. See explanations from get_columns()
            if with_aliases and field.column in col_aliases:
                c_alias = 'Col%d' % len(col_aliases)
                if isinstance(field, models.CharField):
                    result.append('upper(%s.%s) AS %s' % (qn(alias),
                            qn2(field.column), c_alias))
                else:
                    result.append('%s.%s AS %s' % (qn(alias),
                            qn2(field.column), c_alias))
                    
                col_aliases.add(c_alias)
                aliases.add(c_alias)
            else:
                if isinstance(field, models.CharField):
                    r = 'upper(%s.%s)' % (qn(alias), qn2(field.column))
                else:
                    r = '%s.%s' % (qn(alias), qn2(field.column))
                result.append(r)
                aliases.add(r)
                if with_aliases:
                    col_aliases.add(field.column)
        return result, aliases
        
class SQLInsertCompiler(compiler.SQLInsertCompiler, CustomSQLCompiler):
    pass
    
class SQLDeleteCompiler(compiler.SQLDeleteCompiler, CustomSQLCompiler):
    pass

class SQLUpdateCompiler(compiler.SQLUpdateCompiler, CustomSQLCompiler):
    pass

class SQLAggregateCompiler(compiler.SQLAggregateCompiler, CustomSQLCompiler):
    pass

class SQLDateCompiler(compiler.SQLDateCompiler, CustomSQLCompiler):
    pass