# This file contains middleware to update request.META when running behind
# multiple reverse proxis. Django cannot cope with this, and fails to generate
# the correct URIs, particularly the host name.

class MultipleProxyMiddleware(object):
    
    FORWARDED_FOR_FIELDS = [ 
      'HTTP_X_FORWARDED_FOR',
      'HTTP_X_FORWARDED_HOST',
      'HTTP_X_FORWARDED_SERVER',
    ]

    def process_request(self, request):
        """
        When a reverse proxy forwards on a request to a backend server, the proxy
        adds/rewrites some of the headers so that the next server knows it was proxied.
        Eg, after going through proxy1:
            X-Forwarded-Host: proxy1
        If it then goes through an additional proxy, the fields are updated, to be a 
        list of the proxies it has travelled through.
        Eg, after then going through proxy2:
            X-Forwarded-Host: proxy1, proxy2
        Django cannot handle this, so this middleware rewrites these proxy headers
        so that only the most recent proxy is mentioned.
        """
        for field in self.FORWARDED_FOR_FIELDS:
            if field in request.META:
                if ',' in request.META[field]:
                    parts = request.META[field].split(',')
                    request.META[field] = parts[-1].strip()

