"""
Israeli-specific form helpers
"""

from django.forms import ValidationError
from django.forms.fields import RegexField, Field
from django.utils.translation import ugettext_lazy as _
import re

id_number = re.compile(r'^(?P<number>\d{1,8})(?P<check>\d)$')

class ILPostalCodeField(RegexField):
    """
    A form field that validates its input as an Israeli postal code.
    Valid form is XXXXX where X represents integer.
    """
    default_error_messages = {
        'invalid': _(u'Enter a postal code in the format XXXXX'),
    }

    def __init__(self, *args, **kwargs):
        super(ILPostalCodeField, self).__init__(r'^\d{5}$',
            max_length=None, min_length=None, *args, **kwargs)

    def clean(self, value):
        """
        Validates the input and returns a string that contains only numbers.
        Returns an empty string for empty values.
        """
        return super(ILPostalCodeField, self).clean(value)

class ILIDNumberField(Field):
    """
    Israeli identification number field.
    """
    default_error_messages = {
        'invalid': _(u'Enter a valid ID number.'),
    }

    def clean(self, value):
        super(ILIDNumberField, self).__init__(value)

        if value in EMPTY_VALUES:
            return u''

        match = re.match(id_number, value)
        if not match:
            raise ValidationError(self.error_messages['invalid'])

        number, check = match.groupdict()['number'].zfill(8), int(match.groupdict()['check'])

        sum = 0
        weight = 1
        for digit in number+str(check):
            sum += (lambda x: x/10 + x%10)(int(digit)*weight)
            weight ^= 3
                
        if sum%10 != 0:
            raise ValidationError(self.error_messages['invalid'])

