diff --git a/django/contrib/localflavor/au/forms.py b/django/contrib/localflavor/au/forms.py
index 87e0438..1b094e7 100644
--- a/django/contrib/localflavor/au/forms.py
+++ b/django/contrib/localflavor/au/forms.py
@@ -12,15 +12,22 @@ import re
 PHONE_DIGITS_RE = re.compile(r'^(\d{10})$')
 
 class AUPostCodeField(RegexField):
-    """Australian post code field."""
+    """ Australian post code field.
+
+    Assumed to be 4 digits.
+    Northern Territory 3-digit postcodes should have leading zero.
+    """
     default_error_messages = {
-        'invalid': _('Enter a 4 digit post code.'),
+        'invalid': _('Enter a 4 digit postcode.'),
     }
 
     def __init__(self, *args, **kwargs):
+        if 'max_length' in kwargs:
+            kwargs.pop('max_length')
         super(AUPostCodeField, self).__init__(r'^\d{4}$',
-            max_length=None, min_length=None, *args, **kwargs)
+            max_length=4, min_length=None, *args, **kwargs)
 
+        
 class AUPhoneNumberField(Field):
     """Australian phone number field."""
     default_error_messages = {
@@ -40,6 +47,7 @@ class AUPhoneNumberField(Field):
             return u'%s' % phone_match.group(1)
         raise ValidationError(self.error_messages['invalid'])
 
+    
 class AUStateSelect(Select):
     """
     A Select widget that uses a list of Australian states/territories as its
diff --git a/django/contrib/localflavor/au/models.py b/django/contrib/localflavor/au/models.py
new file mode 100644
index 0000000..f2f6e47
--- /dev/null
+++ b/django/contrib/localflavor/au/models.py
@@ -0,0 +1,43 @@
+from django.conf import settings
+from django.utils.translation import ugettext_lazy as _
+from django.db.models.fields import CharField
+from django.contrib.localflavor.au.au_states import STATE_CHOICES
+from django.contrib.localflavor.au import forms
+
+class AUStateField(CharField):
+
+    description = _("Australian State")
+
+    def __init__(self, *args, **kwargs):
+        kwargs['choices'] = STATE_CHOICES
+        kwargs['max_length'] = 3
+        super(AUStateField, self).__init__(*args, **kwargs)
+
+
+class AUPostCodeField(CharField):
+
+    description = _("Australian Postcode")
+
+    def __init__(self, *args, **kwargs):
+        kwargs['max_length'] = 4
+        super(AUPostCodeField, self).__init__(*args, **kwargs)
+
+    def formfield(self, **kwargs):
+        defaults = {'form_class': forms.AUPostCodeField}
+        defaults.update(kwargs)
+        return super(AUPostCodeField, self).formfield(**defaults)
+        
+        
+class AUPhoneNumberField(CharField):
+
+    description = _("Australian Phone number")
+
+    def __init__(self, *args, **kwargs):
+        kwargs['max_length'] = 20
+        super(AUPhoneNumberField, self).__init__(*args, **kwargs)
+
+    def formfield(self, **kwargs):
+        defaults = {'form_class': forms.AUPhoneNumberField}
+        defaults.update(kwargs)
+        return super(AUPhoneNumberField, self).formfield(**defaults)
+    
diff --git a/docs/ref/contrib/localflavor.txt b/docs/ref/contrib/localflavor.txt
index f54341e..699be68 100644
--- a/docs/ref/contrib/localflavor.txt
+++ b/docs/ref/contrib/localflavor.txt
@@ -202,6 +202,21 @@ Australia (``au``)
     A ``Select`` widget that uses a list of Australian states/territories as its
     choices.
 
+.. class:: au.models.AUPhoneNumberField
+
+    A model field that checks that the value is a valid Australian phone
+    number (ten digits).
+
+.. class:: au.models.AUStateField
+
+    A model field that forms represent as a ``forms.AUStateField`` field and
+    stores the three-letter Australian state abbreviation in the database.
+
+.. class:: au.models.AUPostCodeField
+
+    A model field that forms represent as a ``forms.AUPostCodeField`` field 
+    and stores the four-digit Australian postcode in the database.
+
 Austria (``at``)
 ================
 
diff --git a/tests/regressiontests/localflavor/au/__init__.py b/tests/regressiontests/localflavor/au/__init__.py
new file mode 100644
index 0000000..e69de29
diff --git a/tests/regressiontests/localflavor/au/forms.py b/tests/regressiontests/localflavor/au/forms.py
new file mode 100644
index 0000000..e65b924
--- /dev/null
+++ b/tests/regressiontests/localflavor/au/forms.py
@@ -0,0 +1,7 @@
+from django.forms import ModelForm
+from models import AustralianPlace
+
+class AustralianPlaceForm(ModelForm):
+    """ Form for storing an Australian place. """
+    class Meta:
+        model = AustralianPlace
diff --git a/tests/regressiontests/localflavor/au/models.py b/tests/regressiontests/localflavor/au/models.py
new file mode 100644
index 0000000..8b8b103
--- /dev/null
+++ b/tests/regressiontests/localflavor/au/models.py
@@ -0,0 +1,14 @@
+from django.db import models
+from django.contrib.localflavor.au.models import AUStateField, AUPostCodeField
+ 
+class AustralianPlace(models.Model):
+    state = AUStateField(blank=True)
+    state_required = AUStateField()
+    state_default = AUStateField(default="NSW", blank=True)
+    postcode = AUPostCodeField(blank=True)
+    postcode_required = AUPostCodeField()
+    postcode_default = AUPostCodeField(default="2500", blank=True)
+    name = models.CharField(max_length=20)
+    
+    class Meta:
+        app_label = 'localflavor'
diff --git a/tests/regressiontests/localflavor/au/tests.py b/tests/regressiontests/localflavor/au/tests.py
new file mode 100644
index 0000000..05301ea
--- /dev/null
+++ b/tests/regressiontests/localflavor/au/tests.py
@@ -0,0 +1,57 @@
+from django.test import TestCase
+from forms import AustralianPlaceForm
+import re
+ 
+SELECTED_OPTION_PATTERN = r'<option value="%s" selected="selected">'
+BLANK_OPTION_PATTERN = r'<option value="">'
+INPUT_VALUE_PATTERN = r'<input[^>]*value="%s"[^>]*>'
+
+
+class AULocalflavorTests(TestCase):
+    def setUp(self):
+        self.form = AustralianPlaceForm(
+            {'state':'WA',
+             'state_required':'QLD',
+             'name':'dummy',
+             'postcode':'1234',
+             'postcode_required':'4321',
+             })
+        
+    def test_get_display_methods(self):
+        """ Ensure get_*_display() methods are added to model instances. """
+        place = self.form.save()
+        self.assertEqual(place.get_state_display(), 'Western Australia')
+        self.assertEqual(place.get_state_required_display(), 'Queensland')
+
+    def test_default_values(self):
+        """ Ensure that default values are selected in forms. """
+        form = AustralianPlaceForm()
+        self.assertTrue(re.search(SELECTED_OPTION_PATTERN % 'NSW',
+                                  str(form['state_default'])))
+        self.assertTrue(re.search(INPUT_VALUE_PATTERN % '2500',
+                                  str(form['postcode_default'])))
+        
+    def test_required(self):
+        """ Test that required AUStateFields throw appropriate errors. """
+        form = AustralianPlaceForm({'state':'NSW', 'name':'Wollongong'})
+        self.assertFalse(form.is_valid())
+        self.assertEqual(
+            form.errors['state_required'], [u'This field is required.'])
+        self.assertEqual(
+            form.errors['postcode_required'], [u'This field is required.'])        
+    
+    def test_field_blank_option(self):
+        """ Test that the empty option is there. """
+        self.assertTrue(re.search(BLANK_OPTION_PATTERN,
+                                  str(self.form['state'])))
+
+    def test_selected_values(self):
+        """ Ensure selected states match the initial values provided. """
+        self.assertTrue(re.search(SELECTED_OPTION_PATTERN % 'WA',
+                                  str(self.form['state'])))
+        self.assertTrue(re.search(SELECTED_OPTION_PATTERN % 'QLD',
+                                  str(self.form['state_required'])))
+        self.assertTrue(re.search(INPUT_VALUE_PATTERN % '1234',
+                                  str(self.form['postcode'])))
+        self.assertTrue(re.search(INPUT_VALUE_PATTERN % '4321',
+                                  str(self.form['postcode_required'])))
diff --git a/tests/regressiontests/localflavor/tests.py b/tests/regressiontests/localflavor/tests.py
index 85e91a0..e22fc0f 100644
--- a/tests/regressiontests/localflavor/tests.py
+++ b/tests/regressiontests/localflavor/tests.py
@@ -3,3 +3,4 @@ from django.utils import unittest
 
 # just import your tests here
 from us.tests import *
+from au.tests import *
