from django.db.models import AutoField
from django.db.models import ForeignKey

def save_with_refresh(obj):
    """
https://code.djangoproject.com/ticket/10811

Refreshes all foreign keys. Should be used before saving to avoid the
problem below

child = Child.create()
parent = Parent.create()
parent.child = child
child.save()                pk set to 123
parent.child.id             result properly set to 123
parent.save()               django incorrectly stores NULL for parent.child

"""

    # attname.......Presents the property of this model in which field the 
    #               duplicated primary key is saved in the database. 
    # name..........Presents the reference to the object where the original 
    #               primary key is saved. 
    fks = [(f.attname, f.name) for f in obj._meta.local_fields if type(f) == ForeignKey] 
    
    for (attname, fk_ref) in fks: 
        # Update attribute if it is defined but not yet stored as an id
        if not getattr(obj, attname) and getattr(obj, fk_ref): 
    
            # raise an error if a primary key of the related object does 
            # not exist (if it is not saved until now) 
            if getattr(obj, fk_ref).pk == None: 
                raise ValueError("Cannot find a primary key for " + str(fk_ref) + 
                    " as a foreign key in an instance of " + obj.__class__.__name__ + 
                    ". Have you already saved the " + str(fk_ref) + " object?")
    
            else:
                # force django to reset id
                setattr(obj, fk_ref, getattr(obj, fk_ref))

    obj.save()
