#!/usr/bin/env python

###################################################################################################
# Scaffolding
# Author: Javier Nievas (javinievas@gmail.com)
# Date: 6th August 2006
###################################################################################################
# BASED ON...
# Script based on the work of: 
#  - Lllama
#  - Akaihola 
#  - David Warry
###################################################################################################
# INTRODUCTION...
# The original work lives in: http://code.djangoproject.com/wiki/ScaffoldScript
# I have put together the generation of forms and manipulators in the same file
###################################################################################################
# 
# If you choose to use custom templates (-t custom) this script suppose you have three files:
#       templates/scaffold/form.html      - the main template, it contains the form tag and a {{ formfields }}
#       templates/scaffold/pre_form.html  - this file is prepended to generated template before the form
#       templates/scaffold/post_form.html - this file is appened to generated template after the form
# 
# This script was tested under Windows XP, using Django 0.96-pre
# You may have to change some code to match your own installation
###################################################################################################

from optparse import OptionParser
import sys
import os
from django.shortcuts import render_to_response
from django.template.loader import get_template
from django.db.models import get_model
from django.db import models
from django.template import Context, Template



def readfile(file):
    f = open(file,"r")
    pre = f.read()
    f.close()
    
    return pre


def argValue(s, args):
    if s == 'related_query_name':
        return "???"
    else:
        return args[s]
        
def field_text(field):
    d = field.__dict__
    args = dict([(f, isinstance(d[f], str) and "'%s'" % d[f] or d[f]) for f in d if f not in ['db_index', 'primary_key', 'rel', 'core', 'creation_counter',
    'editable', 'name', 'default', 'column', 'attname', 'auto_now_add', 'help_text', 'auto_now', 'unique', 'verbose_name']])
    if args['blank'] is True or args['null'] is True:
        args['is_required'] = False
    else:
        args['is_required'] = True
    try:
        del(args['blank'])
        del(args['null'])
    except KeyError:
        pass
    try:
        args['validator_list'] = ['%s' % v.__name__ for v in args['validator_list']]
    except KeyError:
        pass
    if field.__class__.__name__ == 'CharField':
        fieldType = 'TextField'
    if field.__class__.__name__ == 'TextField':
        fieldType = 'LargeTextField'
    elif field.__class__.__name__ == 'ForeignKey':
        fieldType = 'SelectField'
        args['choices'] = "[('','-------')] + [(o.id, o) for o in %s.get_list()]" % (field.name.lower() + "s",)
    elif field.__class__.__name__ == 'BooleanField':
        fieldType = 'CheckboxField'
    elif field.__class__.__name__ == 'DateTimeField':
        fieldType = 'DatetimeField'
    elif field.__class__.__name__ == 'ManyToManyField':
        fieldType = 'SelectMultipleField'
        args['choices'] = "[(o.id, o) for o in %s.get_list()]" % (field.name.lower() + "s",)
    else:
        fieldType = field.__class__.__name__
    
    extra_args = ["%s=%s" % (s, argValue(s, args)) for s in args if args[s] not in [None, '', False, []]]
    #print field.name, extra_args
    if extra_args:        
        return '''\
            forms.%s(field_name='%s', %s),''' % (fieldType, field.name, ', '.join(extra_args))
    else:
        return '''\
            forms.%s(field_name='%s'),''' % (fieldType, field.name)

def value_text(field):
    return """\
                %s=new_data['%s']""" % (field.name, field.name)

def m2m_text(field):
    return """        temp.set_%s(newdata['%s'])""" % (field.name, field.name)

def image_field_html(name):
    return '''\
        {{ form.NAME_file }} \
        {{ form.NAME }}
        {% if form.NAME.errors %}
        <div class='div_errors'>
        {{ form.NAME.errors|join:", " }}
        </div>
        {% endif %}'''.replace('NAME', name)

def field_html(name):
    return '''\
    {{ form.NAME }}
    {% if form.NAME.errors %}
      <div class='div_errors'>
      {{ form.NAME.errors|join:", " }}
      </div>
    {% endif %}'''.replace('NAME', name)


def labeled_field_html(field):
    if isinstance(field, models.DateTimeField):
        rows = [field_html(field.name + '_' + suffix)
                for suffix in 'date', 'time']
    elif isinstance(field, models.ImageField):
        rows = [image_field_html(field.name)]
    else:
        rows = [field_html(field.name)]
    return '''\
  <li>
    <label for="id_%s">%s:</label>
%s
  </li>''' % (field.name, field.verbose_name or field.name, '\n'.join(rows))

def renderForm(appname, modelname):
    
    modelobj = get_model(appname, modelname)
    
    template = Template(
"""<form method="POST" action=".">
<fieldset>
<legend>{{modelname}}</legend>
<ul>
{{ formfields }}
</ul>
</fieldset>
<input type="submit" value="submit" />
</form>
""")

    context = Context({'formfields': '\n'.join(labeled_field_html(f) for f in modelobj._meta.fields + modelobj._meta.many_to_many if f.name !='id'), 'modelname':modelname})
    
    if (options.template == "custom"):
        t=get_template('scaffold/form.html')
        out = readfile("../apps/%s/templates/scaffold/pre_form.html" % appname )
        out += t.render(context)
        out += readfile("../apps/%s/templates/scaffold/post_form.html" % appname )
        return out
    else:
        return template.render(context)

def renderManipulator(appname, modelname, manipname):
    modelobj = get_model(appname, modelname)
    
    template = Template(
"""class {{ name }}Manipulator(forms.Manipulator):
    def __init__(self, pk=None):
        if pk:
            # change
            self.original_object = {{model}}.objects.get(id=pk)
            self.pk = pk
        else:
            # add
            self.original_object = None
            self.pk = None
        
        self.fields = (
{{ fields }}
        )

    def save(self, new_data):
        if self.original_object:
            # update
            temp = dict(
{{ values }}            
            )
            for k,v in temp.iteritems():
                self.original_object.__setattr__(k, v)
            self.original_object.save()            
            return {{model}}.objects.get(id=pk)
            
        else:
            # insert
            temp = {{ model }}(
{{ values }}
            )
{{ m2m }}
            temp.save()
            return temp
        """)

    context = Context({'fields': '\n'.join([field_text(f) for f in modelobj._meta.fields + modelobj._meta.many_to_many if f.name !='id']), 
                'name': manipname, 
                'model': modelname, 
            'values': ',\n'.join([value_text(f) for f in modelobj._meta.fields if f.name != 'id']),                                
                'm2m': '\n'.join([m2m_text(f) for f in modelobj._meta.many_to_many])})

    if (options.template == "custom"):
        t=get_template('scaffold/manipulator.py')
        return  t.render(context)
    else:
        return template.render(context)


if __name__ == "__main__":

    try:
        import settings
    except ImportError:
        print "Settings file not found.  Place this file in the same place as manage.py"
        sys.exit()

    project_directory = os.path.dirname(settings.__file__)
    project_name = os.path.basename(project_directory)
    sys.path.append(os.path.join(project_directory, '..'))
    project_module = __import__(project_name, '', '', [''])
    sys.path.pop()
    os.environ['DJANGO_SETTINGS_MODULE'] = '%s.settings' % project_name

    p = OptionParser()

    #p.add_option("-p", "--project", dest="project", help="The project which contains the model")
    p.add_option("-g","--generate", dest="generate", help="Choose between form or manipulator")
    p.add_option("-a", "--app", dest="app", help="The app which contains the model")
    p.add_option("-m", "--model", dest="model", help="The model to produce the form for")
    p.add_option("-n", "--name", dest="name", help="The name of the custom manipulator")
    p.add_option("-t", "--template", dest="template", help="Type custom to use your template/scaffold directory or leave blank for default template")

    options, args = p.parse_args()

    if not (options.generate and options.model and options.app):
        p.print_help()
        sys.exit()

    if not options.name:
        options.name = "%s%s" % (options.app, options.model)

    m = __import__("%s.models" % (options.app,), '', '', [options.model])

    a = getattr(m, options.model)
    
    if options.generate == "form":
        print renderForm(options.app, options.model)
    elif options.generate == "manipulator":
        print renderManipulator(options.app, options.model, options.name)

