class Model(object):
	def __init__(self, table_name):
		self.table_name = table_name
	
	def query(self, *result):
		return Query(self.table_name, result)

class Query(object):
	def __init__(self, table, clauses=None):
		if (clauses):
			self.object = WhereAnd(clauses)
		else:	
			self.object = None
			
		self.results = None
		self.len = None
		self.table = table
		self.parent = None
	
	def __iadd__(self, other):
		"Results from self += other"
		assert self.table == other.table, "Table mismatch in queries."
		assert self.parent == other.parent, "Parent mismatch in queries.  They both aren't subqueries of the same query."
		self.object = WhereOr((self.object, other.object))
		self.results = None	# Since we're modifying this, delete our caches
		self.len = None
		return self
	
	def __add__(self, other):
		"Results from (self + other)"
		assert self.table == other.table, "Table mismatch in queries."
		assert self.parent == other.parent, "Parent mismatch in queries.  They both aren't subqueries of the same query."
		q = Query(self.table)
		q.object = WhereOr((self.object, other.object))
		q.parent = self.parent
		return q
	
	def _get_final_object(self):
		"Resovles the final object in case this Query has a parent (is a sub-query)."
		if (self.parent == None):
			return self.object
		else:
			return WhereAnd((self.object, self.parent.object))
		
	def fetch(self):
		"Hits the database, and returns the results"
		print "SELECT * FROM `%s` WHERE %r" % (self.table, self._get_final_object())
		self.results = ["Bob"]	# Since we don't actually have a database, return Bob.
		return self.results
	
	def _count(self):
		"If we don't actually use the results, and just need the length, this optimizes it a bit."
		print "SELECT COUNT(*) FROM `%s` WHERE %r" % (self.table, self._get_final_object())
		self.len = 1
	
	def sub_query(self, *clauses):
		"Creates a sub-query, all results in that query come from the results of this."
		q = Query(self.table, clauses)
		q.parent = self
		return q
	
	def __iter__(self):
		"Container emulation"
		if (self.results == None):
			self.fetch()
		return self.results.__iter__()
		
	def __len__(self):
		"Container emulation"
		if (self.results):				# If we have results, return the length of those
			return len(self.results)
		elif (self.len == None):		# If we don't have a cache of the length, query the database
			self._count()
		return self.len				# We must have a cache of the length, so return that.
		
	def __getitem__(self, i):
		"Container emulation"
		if (self.results == None):
			self.fetch()
		return self.results[i]
		
	def __contains__(self, o):
		"Container emulation | i.e. if (o in self):"
		if (self.results == None):
			self.fetch()
		return o in self.results
		
	def __str__(self):
		"Container emulation | print self"
		if (self.results):
			return str(self.results)
		else:
			return str(self.fetch())

class WhereOr(object):
	"OR clause"
	def __init__(self, clauses):
		self.clauses = clauses
	
	def __repr__(self):
		return "(%s)" % " OR ".join([repr(c) for c in self.clauses])
		
class WhereAnd(object):
	"AND clause"
	def __init__(self, clauses):
		self.clauses = clauses
	
	def __repr__(self):
		return "(%s)" % " AND ".join([repr(c) for c in self.clauses])

class WhereClause(object):
	"Operator clause | i.e. field < value"
	def __init__(self, *args):
		self.args = args
	
	def __repr__(self):
		return "%s %s %r" % self.args
		
class Field(object):
	def __init__(self, name):
		self.name = name
	
	def __lt__(self, other):
		return WhereClause(self.name, "<", other)
		
	def __gt__(self, other):
		return WhereClause(self.name, ">", other)
		
	def __eq__(self, other):
		return WhereClause(self.name, "=", other)
		
	def __ne__(self, other):
		return WhereClause(self.name, "<>", other)

	def __le__(self, other):
		return WhereClause(self.name, "<=", other)
		
	def __ge__(self, other):
		return WhereClause(self.name, ">=", other)
	
	def contains(self, other):
		return WhereClause(self.name, "LIKE", "%%%s%%" % other)
	
	def startswith(self, other):
		return WhereClause(self.name, "LIKE", "%s%%" % other)
	
	def endswith(self, other):
		return WhereClause(self.name, "LIKE", "%%%s" % other)
	
class FieldFactory(object):
	def __getattr__(self, k):
		return Field(k)
	
	def __getitem__(self, i):
		return Field(i)

# These would be imported, as in: 
#     django.models.app import polls, choices
#     django.database import field
field = FieldFactory()
polls = Model('polls')
choices = Model('choices')


# The results will always be ['Bob'], because we aren't actually talking to a database.

# Basic union:
results  = choices.query(field.choice == 'A choice')
results += choices.query(field.votes > 4)
print results[0]    # Database hit

# Sub-query:
results = choices.query(field.choice == 'A choice')
popular_choices = results.sub_query(field.votes > 10)
print popular_choices  # Database hit

# Equivelant to the Sub-query
popular_choices = choices.query(field.choice == 'A choice', field.votes > 10)
print popular_choices  # Database hit

# LIKE / Startswith
obstaining = choices.query(field.choice.startswith("I don't know"))
print obstaining

# Never called
never_results = choices.query(field.choice == 'There is never results, cause I am never used.')

# Fetched
results  = choices.query(field.choice == 'A choice')
results.fetch()  # Database hit
print results  # Cache used

# Only counted
results  = choices.query(field.choice == 'A choice')
print len(results)  # Database hit, but only with a count

# Uncomment to create an Assertion error, because we're adding two different table results
#results = polls.query(field.question == 'Love to error out!')
#results += choices.query(field.choice == 'A choice and a question?  Throw Assertion!')

