# -*- coding: utf-8 -*-
"""
Bulgarian specific Form helpers

@AUTHOR: Jordan Dimov (jdimov@a115.bg)
"""

import re

from django.forms import ValidationError, Field, RegexField, Select
from django.forms.fields import EMPTY_VALUES
from django.utils.translation import ugettext_lazy as _
from django.utils.encoding import smart_unicode

class BGPostalCodeField(RegexField):
	"""
	A Bulgarian postal code field.  

	Validates that the value is a four-digit number greater than or equal to 1000.  
	"""
	default_error_messages = { 
			'invalid': _('Enter a valid postal code in the format XXXX'), 
			}
	def __init__(self, *args, **kwargs):
		super(BGPostalCodeField, self).__init__(r'^[1-9][0-9]{3}$', max_length=4, min_length=4, *args, **kwargs)

class BGEGNField(RegexField):
	"""
	A Bulgarian EGN (Social Security) number.

	Validates that the value is a 10-digit EGN number with valid checksum. 
	"""
	default_error_messages = { 
			'invalid': _('Enter a valid 10-digit EGN number.'), 
			}
	def __init__(self, *args, **kwargs):
		super(BGEGNField, self).__init__(r'^[0-9]{10}$', max_length=10, min_length=10, *args, **kwargs)
	def clean(self, value):
		super(BGEGNField, self).clean(value)
		if value in EMPTY_VALUES: 
			return u'' 
		d = (2*int(value[0]) + 4*int(value[1]) + 8*int(value[2]) + 5*int(value[3]) + 10*int(value[4]) + 9*int(value[5]) + 7*int(value[6]) + 3*int(value[7]) + 6*int(value[8])) % 11
		if d != 10:
			if d != int(value[9]):
				raise ValidationError(self.error_messages['invalid'])
		else:
			if d != 0:
				raise ValidationError(self.error_messages['invalid'])
		return value

phone_digits_re = re.compile(r'^(0|\+?359)(\d+)$')
class BGPhoneNumberField(Field): 
	"""
	A Bulgarian phone number field.

	Validates that the number begins with the country code (+359) or with a zero.  Allowed characters are digits, dashes (-), parentheses and spaces (and the leading plus sign).  

	Normalizes to "+359" followed by digits only.  
	"""
	default_error_messages = { 
			'invalid': _('Phone numbers must start with +359 or with 0.'), 
			} 
	def clean(self, value): 
		super(BGPhoneNumberField, self).clean(value) 
		if value in EMPTY_VALUES: 
			return u'' 
		value = re.sub('(\(|\)|\s+|\-)', '', smart_unicode(value)) 
		m = phone_digits_re.search(value) 
		if m: 
			return u'+359%s' % (m.group(2))
		raise ValidationError(self.error_messages['invalid'])

class BGCountySelect(Select): 
	""" 
	A Select widget that uses a list of Bulgarian counties as its choices. 
	""" 
	def __init__(self, attrs=None): 
		from bg_cities import BG_COUNTIES 
		super(BGCountySelect, self).__init__(attrs, choices=BG_COUNTIES)
