from email import Encoders
from email.MIMEBase import MIMEBase
from email.Utils import formatdate

from django.core.mail import make_msgid, SafeMIMEText, SafeMIMEMultipart
from django.core.mail import EmailMessage
from django.conf import settings
from django.utils.encoding import smart_str

class EmailAlternativesMessage(EmailMessage):
    def __init__(self, subject='', body='', from_email=None, to=None, bcc=None,
            connection=None, attachments=None, headers=None, alternatives=None):
        super(EmailAlternativesMessage, self).__init__(subject, body, 
                from_email, to, bcc, connection, attachments, headers)
        self.alternatives=alternatives or []

    def message(self):
        encoding = self.encoding or settings.DEFAULT_CHARSET
        msg = SafeMIMEText(smart_str(self.body, settings.DEFAULT_CHARSET),
                           self.content_subtype, encoding)
        if self.alternatives:
            body_msg = msg
            msg = SafeMIMEMultipart(_subtype='alternative')
            if self.body:
                msg.attach(body_msg)
            for alternative in self.alternatives:
                if isinstance(alternative, MIMEBase):
                    msg.attach(alternative)
                else:
                    msg.attach(self._create_alternative(*alternative))
        if self.attachments:
            body_msg = msg
            msg = SafeMIMEMultipart(_subtype=self.multipart_subtype)
            if self.body:
                msg.attach(body_msg)
            for attachment in self.attachments:
                if isinstance(attachment, MIMEBase):
                    msg.attach(attachment)
                else:
                    msg.attach(self._create_attachment(*attachment))
        msg['Subject'] = self.subject
        msg['From'] = self.from_email
        msg['To'] = ', '.join(self.to)
        msg['Date'] = formatdate()
        msg['Message-ID'] = make_msgid()
        for name, value in self.extra_headers.items():
            msg[name] = value
        return msg

    def attach_alternative(self, content=None, mimetype=None):
        """Attach an alternative content representation."""
        assert content is not None
        assert mimetype is not None
        self.alternatives.append((content, mimetype))

    def _create_alternative(self, content, mimetype):
        """
        Converts the content, mimetype pair into a MIME attachment object.
        """
        basetype, subtype = mimetype.split('/', 1)
        if basetype == 'text':
            alternative = SafeMIMEText(smart_str(content,
                settings.DEFAULT_CHARSET), subtype, settings.DEFAULT_CHARSET)
        else:
            alternative = MIMEBase(basetype, subtype)
            alternative.set_payload(content)
            Encoders.encode_base64(attachment)
        return alternative

