"""Proof of concept for on-the-fly, standalone DJango filter code.

Problem: in some of my work I use the nifty DJango template code
  for documentation generation in standalone code - i.e. I generate 
  documentation of some form without using any web app/infrastructure 
  at all.
  
  DJango can works fine sans-web so long as you set it up correctly
  and you don't use custom filters or tags.
  
  Actually, thats not 100% true. You *can* use custom filters and tags,
  but it takes a bit of magic that I didn't think should be necessary.
  
  The challenge was to submit a patch to the DJango code, but to do
  that I need a small demo app that proves the new work..


Objective: Render a template, completely standalone, using custom filters
  and tags, in a "Django-compliant" way with minimal code changes to DJango.
  
  Specifically, this demo renders some restructured text using filters
  to generate underlines and header formatting.
  
Remember: the changes to DJango mustn't break any existing behavior!

A. Roark, June 2007
"""
import sys, re, os, os.path

try: import django
except ImportError, e:
	sys.stderr.write("DJango not installed\n")
	sys.exit(1)

template_content="""{{ title|rst_format_title:"=" }}

{{author|rst_format_author}}
{% rst_doc_date %}

{{subtitle|rst_format_h1}}

{{para}}
"""

def do_filter_demo():
	"""Render a template from a string using our local filter.
	
	This will render something like:
	>>>
	===============
	 Filtered Demo
	===============
	
	:Author: A. Roark
	:Date:   2007-06-30 21:45:40.205000
	
	Simple restructured text demo
	-----------------------------
	
	This uses builtin filters for formatting text
	<<<
"""
	
	# Render the template here - note that no special imports/code needed.
	t=Template(template_content)
	context=Context({
		'title':   'Filtered Demo',
		'author':   'A. Roark',
		'subtitle': 'Simple restructured text demo',
		'para':     'This uses builtin filters for formatting text', 
	})
	out=t.render(context)
	print "\nDjango Custom Filter Demo:\n>>>\n%s\n<<<\n" % out


#
# --- Define some filters here ---
#

def filter_rst_h1(value, arg='-'):
	"Write a single underline under the text"
	line=arg * len(value)
	return "%s\n%s" % (value, line)

def filter_rst_title(value, arg='-'):
	"Add a line before and after the text"
	line=arg * (2+len(value))
	return "%s\n %s \n%s" % (line, value, line)

def filter_rst_author(value, arg=None):
	return ":Author: %s" %value

#
# --- A simple tag --- 
#
def tag_rst_doc_date(parser, token):
	# A "real" tag should be much more than this - 
	# I'm amazed this works as-is, but its ok for a demo.
	# For real details, see:
	# http://www.djangoproject.com/documentation/templates_python/#writing-the-compilation-function
	import datetime
	return ":Date:   %s" % datetime.datetime.now()

#
# --- Main code ---
#
if __name__ == "__main__":

	# In a standalone app, do this first, before other imports
	from django.conf import settings
	settings.configure( 
		# for debug:
		#   DEBUG=True, TEMPLATE_DEBUG=True,
		# for template files, something like...
		#   TEMPLATE_DIRS=(root_dir, os.path.join(root_dir, 'ref'))
		)

	from django.template import Template, Context, BuiltinLibrary
	
	# set up extra library of routines for later use...
	custom_library = BuiltinLibrary()
	custom_library.filter('rst_format_title',  filter_rst_title)
	custom_library.filter('rst_format_h1',     filter_rst_h1)
	custom_library.filter('rst_format_author', filter_rst_author)
	custom_library.tag(   'rst_doc_date',      tag_rst_doc_date)
	
	do_filter_demo()

