import unittest

from django.http import HttpRequest, HttpResponse

from middleware import SignedCookiesMiddleware

class SignedCookiesTestCase(unittest.TestCase):

    def setUp(self):
        self.middleware = SignedCookiesMiddleware()
        (self.key, self.value) = ('key', 'value')
        self.signature = self.middleware.get_digest(self.key, self.value)
        self.signed_value = '%s:%s' % (self.signature, self.value)

    def test_unsigned_cookie(self):
        "Make sure process_request removes unsigned cookies"
        request = HttpRequest()
        request.COOKIES[self.key] = self.value
        self.middleware.process_request(request)
        self.assert_(self.key not in request.COOKIES)

    def test_invalid_cookie(self):
        "Make sure process_request removes invalid cookies"
        request = HttpRequest()
        request.COOKIES[self.key] = self.signed_value + 'test'
        self.middleware.process_request(request)
        self.assert_(self.key not in request.COOKIES)

    def test_valid_cookie(self):
        "Make sure process_request removes signatures from valid cookies"
        request = HttpRequest()
        request.COOKIES[self.key] = self.signed_value
        self.middleware.process_request(request)
        self.assertEqual(request.COOKIES[self.key], self.value)

    def test_set_cookie(self):
        "Make sure process_response signs cookies appropriately"
        response = HttpResponse()
        response.set_cookie(self.key, self.value)
        self.middleware.process_response(HttpRequest(), response)
        self.assertEqual(response.cookies[self.key].value, self.signed_value)

    def test_delete_cookie(self):
        "Make sure process_response leaves deleted cookies alone"
        response = HttpResponse()
        response.delete_cookie(self.key)
        self.middleware.process_response(HttpRequest(), response)
        self.assertEqual(response.cookies[self.key].value, '')