"Memcached cache backend"

from django.core.cache.backends.base import BaseCache, InvalidCacheBackendError

try:
    import memcache
except ImportError:
    raise InvalidCacheBackendError, "Memcached cache backend requires the 'memcache' library"

#
# The memcache module refuses to use keys longer than 250 characters or containing 
# "control characters", which it defines as anything with a decimal ASCII code below 
# 33. This definition includes the non-printable ASCII characters and the space character, 
# so to allow caching of very long URLs or URLs containing spaces we'll hash them.
#
try:
    import hashlib
    key_hasher = hashlib.sha1
except ImportError:
    import sha
    key_hasher = sha.new

class CacheClass(BaseCache):
    def __init__(self, server, params):
        BaseCache.__init__(self, params)
        self._cache = memcache.Client(server.split(';'))

    def hash_key(self, key):
        if ' ' not in key and len(key) < memcache.SERVER_MAX_KEY_LENGTH:
            return key

        key_hash = key_hasher()
        key_hash.update(key)
        return key_hash.hexdigest()

    def get(self, key, default=None):
        val = self._cache.get(self.hash_key(key))
        if val is None:
            return default
        else:
            return val

    def set(self, key, value, timeout=0):
        self._cache.set(self.hash_key(key), value, timeout or self.default_timeout)

    def delete(self, key):
        self._cache.delete(self.hash_key(key))

    def get_many(self, keys):
        keys = [self.hash_key(key) for key in keys]
        return self._cache.get_multi(keys)
