"""
``db_table``             The name of the table to create for storing the m2m data. 
                         If this isn't given, Django will use ``app_label + '_' + table1 + '_' + table2``. 
"""

from django.db import models

class Publication(models.Model):
    title = models.CharField(maxlength=30)

    def __str__(self):
        return self.title

    class Meta:
        ordering = ('title',)

class Article(models.Model):
    headline = models.CharField(maxlength=100)
    publications = models.ManyToManyField(Publication, db_table='newname')

    def __str__(self):
        return self.headline

    class Meta:
        ordering = ('headline',)

__test__ = {'API_TESTS':"""
# Create a couple of Publications.
>>> p1 = Publication(id=None, title='The Python Journal')
>>> p1.save()
>>> p2 = Publication(id=None, title='Science News')
>>> p2.save()

>>> p1.id
1

# Should be able to query all of the Publications
>>> Publication.objects.all()
[<Publication: Science News>, <Publication: The Python Journal>]

# Create a couple of Articles.
>>> a1 = Article(id=None, headline='Django lets you build Web apps easily')
>>> a1.save()

>> a1.id
1

# Associate the Article with some Publications.
>>> a1.publications.add(p1)
>>> a1.publications.add(p2)

# Should be able to get all of the related publications
>>> a1.publications.all()
[<Publication: Science News>, <Publication: The Python Journal>]

>>> Article.objects.filter(headline__exact='Django lets you build Web apps easily')
[<Article: Django lets you build Web apps easily>]

# We can perform kwarg queries across m2m relationships
>>> Article.objects.filter(publications__id__exact='1')
[<Article: Django lets you build Web apps easily>]

# Custom SQL should return correct values still
>>> Article.objects.extra(where=['id IN (1, 2, 3, 4, 5)'])
[<Article: Django lets you build Web apps easily>]

# Custom SQL using the old table name should error
>>> Article.objects.extra(tables=['custom_tables_article_publications'])
Traceback (most recent call last):
...
OperationalError: no such table: custom_tables_article_publications

# Custom SQL using the correct table name should return correct values still
>>> Article.objects.extra(tables=['newname'])
[<Article: Django lets you build Web apps easily>, <Article: Django lets you build Web apps easily>]

# Changing table name must not give new properties to model
>>> a1.newname
Traceback (most recent call last):
...
AttributeError: 'Article' object has no attribute 'newname'

"""}
