class CaseInsensitiveDict(dict):
    def __init__(self, *kw, **kwargs):
        super(CaseInsensitiveDict, self).__init__(*kw, **kwargs)
        # Set up case insensitive dict for future reference.
        keys = {}
        for key in self.keys():
            if isinstance(key, basestring):
                keys[key.lower()] = key
        self._case_insensitive_keys = keys
        
    def _get_case_sensitive_key(self, key):
        if isinstance(key, basestring):
            key = self._case_insensitive_keys.get(key.lower(), key)
        return key

    def __getitem__(self, key):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).__getitem__(key)

    def __setitem__(self, key, value):
        if isinstance(key, basestring):
            key_lower = key.lower()
            if key_lower in self._case_insensitive_keys.keys():
                # Delete old key (since it may be a different case).
                old_key = self._case_insensitive_keys[key_lower]
                super(CaseInsensitiveDict, self).__delitem__(old_key)
            # Save new key in case insensitive dict.
            self._case_insensitive_keys[key_lower] = key
        return super(CaseInsensitiveDict, self).__setitem__(key, value)

    def __delitem__(self, key):
        key = self._get_case_sensitive_key(key)
        if isinstance(key, basestring):
            # Delete key from case insensitive dict.
            del self._case_insensitive_keys[key.lower()]
        return super(CaseInsensitiveDict, self).__delitem__(key)

    def __contains__(self, key):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).__contains__(key)

    def get(self, key):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).get(key)

    def has_key(self, key):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).has_key(key)

    def pop(self, key):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).pop(key)

    def update(self, dict=None, **kwargs):
        if dict is None:
            dict = kwargs
        if dict:
            # Check for keys in new dict which may differ to the current keys.
            for key in dict.keys():
                case_sensitive_key = self._get_case_sensitive_key(key)
                if key != case_sensitive_key:
                    # Change case to match existing key.
                    dict[case_sensitive_key] = dict[key]
                    del dict[key]
        return super(CaseInsensitiveDict, self).update(dict)

    def clear(self):
        # Reset case insensitive dict
        self._case_insensitive_keys = {}
        return super(CaseInsensitiveDict, self).clear()

    def setdefault(self, key, default=None):
        key = self._get_case_sensitive_key(key)
        return super(CaseInsensitiveDict, self).setdefault(key, default)
        
