class PaginatorPage(object):
    """
    This class is used to work with a page of a paginator object.
    Pass a paginator object, and a valid page number.
    It uses the same methods as a paginator, except you do not need
    to use a page number, and also provides some other useful
    properties.
    
    `page_number` is always 0-based to match the Paginator object.
    `base` allows next_page_number, previous_page_number and
    page_number return their numbers using a different base.
    
    For example:
    >>> page = PaginatorPage(paginator, 0, base=1)
    >>> page.next_page_number()
    2
    >>> page.page_number
    1
    """
    def __init__(self, paginator, page_number, base=0):
        self.paginator = paginator
        # _page_number remains the base-0 page number
        self._page_number = paginator.validate_page_number(page_number)
        self.base = int(base)

    def get_page(self):
        if not hasattr(self, '_page'):
            self._page = self.paginator.get_page(self._page_number)
        return self._page

    def has_next_page(self):
        return self.paginator.has_next_page(self._page_number)

    def has_previous_page(self):
        return self.paginator.has_previous_page(self._page_number)

    def first_on_page(self):
        return self.paginator.first_on_page(self._page_number)

    def last_on_page(self):
        return self.paginator.last_on_page(self._page_number)

    def _get_hits(self):
        return self.paginator.hits
    hits = property(_get_hits)

    def _get_pages(self):
        return self.paginator.pages
    pages = property(_get_pages)

    def _get_page_number(self):
        self._page_number + base
    page_number = property(_get_page_number)

    def _get_next_page_number(self):
        if self.has_next_page():
            # Uses page_number so that we return the correct base
            return self.page_number + 1
    next_page_number = property(_get_next_page_number)

    def _get_previous_page_number(self):
        if self.has_previous_page():
            # Uses page_number so that we return the correct base
            return self.page_number - 1
    previous_page_number = property(_get_previous_page_number)
