#!/usr/bin/env python
# -*- coding: utf-8 -*-
"""Miscellaneous IPv4 Utilities"""

## ip utils

def is_valid_ipv4_addr(address):
    """is_valid_ipv4_addr(ip_addr) -> bool
    
    returns True if address is a valid IPv4 address
    
    Examples:

    >>> is_valid_ipv4_addr('asdf')
    False
    >>> is_valid_ipv4_addr('a.b.c.d')
    False
    >>> is_valid_ipv4_addr('2.3.4.5')
    True
    >>> is_valid_ipv4_addr('127.0.0.1')
    True
    >>> is_valid_ipv4_addr('django.py')
    False
    >>> is_valid_ipv4_addr('256.23.2.2')
    False
    >>> is_valid_ipv4_addr('0.0.0.0')
    True
    >>> is_valid_ipv4_addr('25.25.25.25')
    True
    """
    return 4 == (1 + address.count(".")) == \
        len([n for n in address.split(".") 
        if (n.isdigit() and -1 < int(n) < 256)])

def is_valid_ipv4_port(port):
    """is_valid_ipv4_port(port) -> bool
    
    returns True if 'port' is a valid IPv4 port

    Examples:

    Known good ports 
    
    >>> is_valid_ipv4_port(80) 
    True

    Minimum good port
    
    >>> is_valid_ipv4_port(0) 
    True

    Maximum good port
    
    >>> is_valid_ipv4_port(65535) 
    True

    A string
    
    >>> is_valid_ipv4_port('25') 
    True

    Known bad ports 
    
    >>> is_valid_ipv4_port(65536) 
    False
    
    >>> is_valid_ipv4_port(-1) 
    False
    """
    return str(port).isdigit() and -1 < int(port) < 65536

def make_ipv4_host_port(host_port=None, default_host="127.0.0.1", default_port=8000):
    """make_ipv4_host_port([host_port, default_host, default_port]) -> tuple

    returns a valid host/port tuple from string 'host_port'
    
    Examples:
   
    >>> make_ipv4_host_port()
    ('127.0.0.1', 8000)

    >>> make_ipv4_host_port('168.223.2.25')
    ('168.223.2.25', 8000)
    
    >>> make_ipv4_host_port('bad bad')
    ('127.0.0.1', 8000)
    
    >>> make_ipv4_host_port('bad too', default_host='127.0.0.1')
    ('127.0.0.1', 8000)
    
    >>> make_ipv4_host_port('666.77.23.32:1')
    ('127.0.0.1', 1)
    
    >>> make_ipv4_host_port('25.32.15.15:81')
    ('25.32.15.15', 81)
    """
    if not host_port:
        return (default_host, default_port)
    address = host_port.split(":", 1)
    (addr_index, port_index) = (len(address) == 2) and (0, 1) or (0, 0)
    ip_addr = is_valid_ipv4_addr(address[addr_index]) and \
        (address[addr_index]) or \
        default_host
    ip_port = is_valid_ipv4_port(address[port_index]) and \
        (int(address[port_index])) or \
            default_port
    return (ip_addr, ip_port)

#### doctests 
##if __name__ == '__main__':
##    import doctest
##    doctest.testmod()
##
