import uuid

from django.conf import settings
from django.contrib.auth.models import AbstractBaseUser, BaseUserManager, PermissionsMixin
from django.db import models


class Person(PermissionsMixin, AbstractBaseUser):
    email = models.EmailField(max_length=254, unique=True)
    id = models.UUIDField(default=uuid.uuid4, editable=False, primary_key=True)
    first_name = models.CharField(max_length=35)
    is_active = models.BooleanField(default=True)
    is_staff = models.BooleanField(default=False)
    last_name = models.CharField(max_length=35)
    
    objects = BaseUserManager()
    
    USERNAME_FIELD = 'email'
    REQUIRED_FIELDS = ['first_name', 'last_name']
    
    def __str__(self):
        return self.get_full_name()
    
    def get_full_name(self):
        return '%(first_name)s %(last_name)s' % {'first_name': self.first_name, 'last_name': self.last_name}
    
    def get_short_name(self):
        return self.first_name


class Site(models.Model):
    id = models.UUIDField(default=uuid.uuid4, editable=False, primary_key=True)
    members = models.ManyToManyField(settings.AUTH_USER_MODEL, related_name='sites', related_query_name='site')
    title = models.CharField(max_length=150)
    
    def __str__(self):
        return self.title


class Page(models.Model):
    author = models.ForeignKey(settings.AUTH_USER_MODEL, related_name='pages', related_query_name='page')
    id = models.UUIDField(default=uuid.uuid4, editable=False, primary_key=True)
    site = models.ForeignKey('Site', related_name='pages', related_query_name='page')
    title = models.CharField(max_length=150)
    
    def __str__(self):
        return self.title
