import sys

from types import ModuleType

from django.db import models

# db schema and traceback at the bottom

"""
>>> from modeltest.models import get_module
>>> 
>>> m = get_module()
>>> m.A(id=1, name='a').save()
>>> m.B(id=1, name='b').save()
>>> m.AB(a=m.A.objects.get(id=1), b=m.B.objects.all()[0], extra='ab').save()
>>> m.AB.objects.select_related('a', 'b') # works in both 1.6 and 1.7
>>> m.A.objects.get(id=1).b_s.all() # works in 1.6, raises FieldError in 1.7
>>>
"""


class A(models.Model):
    id = models.IntegerField(primary_key=True, db_column='a_primary_id')
    name = models.CharField(max_length=12)
    b_s = models.ManyToManyField('B', through='AB')

    class Meta:
        abstract = True


class B(models.Model):
    id = models.IntegerField(primary_key=True, db_column='b_primary_id')
    name = models.CharField(max_length=12)
    a_s = models.ManyToManyField('A', through='AB')

    class Meta:
        abstract = True

class AB(models.Model):
    a = models.ForeignKey('A', db_column='rel_to_a')
    b = models.ForeignKey('B', db_column='rel_to_b')
    extra = models.CharField(max_length=12, blank=True, null=True)

    class Meta:
        unique_together = (('a', 'b'),)
        abstract = True

        
def get_module():
    name = 'modeltest_dynamic.models'
    
    if name in sys.modules:
        return sys.modules[name]
    
    mod = ModuleType(name)

    mod.A = type('A', (A,), {
        '__module__':mod.__name__,
        'Meta':type('Meta', (A.Meta, object), {
            '__module__':mod.__name__,
            'app_label':'modeltest_dynamic',
            'managed':False,
            'db_table':A._meta.db_table
        })
    })

    mod.B = type('B', (B,), {
        '__module__':mod.__name__,
        'Meta':type('Meta', (B.Meta, object), {
            '__module__':mod.__name__,
            'app_label':'modeltest_dynamic',
            'managed':False,
            'db_table':B._meta.db_table
        })
    })

    mod.AB = type('AB', (AB,), {
        '__module__':mod.__name__,
        'Meta':type('Meta', (AB.Meta, object), {
            '__module__':mod.__name__,
            'app_label':'modeltest_dynamic',
            'managed':False,
            'db_table':AB._meta.db_table
        })
    })
    
    sys.modules[name] = mod
    return mod


"""
CREATE TABLE "modeltest_a" (
    "a_primary_id" integer NOT NULL PRIMARY KEY,
    "name" varchar(12) NOT NULL
);
CREATE TABLE "modeltest_b" (
    "b_primary_id" integer NOT NULL PRIMARY KEY,
    "name" varchar(12) NOT NULL
);
CREATE TABLE "modeltest_ab" (
"id" integer NOT NULL PRIMARY KEY AUTOINCREMENT,
"rel_to_a" integer NOT NULL REFERENCES "modeltest_a" ("a_primary_id"),
"rel_to_b" integer NOT NULL REFERENCES "modeltest_b" ("b_primary_id"),
"extra" varchar(12),
UNIQUE ("rel_to_a", "rel_to_b")
);
CREATE INDEX "modeltest_ab_4d6cc2fb" ON "modeltest_ab" ("rel_to_a");
CREATE INDEX "modeltest_ab_2c14050b" ON "modeltest_ab" ("rel_to_b");
"""
