from django.test import TestCase
from models import Outer, Inner

class ReverseIsNullTests(TestCase):
    """ Test isnull filters on reverse relations. """

    # Tests are in order: filter(isnull=True), filter(isnull=False), 
    # exclude(isnull=True), exclude(isnull=False)
    
    def setUp(self):
        # Create outer objects
        self.outerA = Outer.objects.create(name="A")
        self.outerB = Outer.objects.create(name="B")

    def test_isnull_empty_reverse(self):
        # No Inner-object: expecting [A,B], [], [], [A,B]
        self.assertEquals(list(Outer.objects.filter(inner__isnull=True)), [self.outerA, self.outerB])
        self.assertEquals(list(Outer.objects.filter(inner__isnull=False)), [])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=True)), [])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=False)), [self.outerA, self.outerB])

    def test_isnull_related_reverse(self):
        # Create Inner-object with relation to A
        inner = Inner.objects.create(outer=self.outerA)

        # One related Inner-object to A: expecting [B], [A], [A], [B]
        self.assertEquals(list(Outer.objects.filter(inner__isnull=True)), [self.outerB])
        self.assertEquals(list(Outer.objects.filter(inner__isnull=False)), [self.outerA])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=True)), [self.outerA])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=False)), [self.outerB])

        inner.delete()

    def test_isnull_nonrelated_reverse(self):
        # Create Inner object without relation
        inner = Inner.objects.create()

        # No related Inner-object: expecting [A,B], [], [], [A,B]
        self.assertEquals(list(Outer.objects.filter(inner__isnull=True)), [self.outerA, self.outerB])
        self.assertEquals(list(Outer.objects.filter(inner__isnull=False)), [])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=True)), [])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=False)), [self.outerA, self.outerB])

        inner.delete()

    def test_isnull_reverse(self):
        # Create Inner object without relation and with
        inner   = Inner.objects.create()
        innerR  = Inner.objects.create(outer=self.outerA)

        # One related Inner-object to A, one with no relation: expecting [B], [A], [A], [B]
        self.assertEquals(list(Outer.objects.filter(inner__isnull=True)), [self.outerB])
        self.assertEquals(list(Outer.objects.filter(inner__isnull=False)), [self.outerA])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=True)), [self.outerA])
        self.assertEquals(list(Outer.objects.exclude(inner__isnull=False)), [self.outerB])

        inner.delete()
        innerR.delete()


