"""
Firebird database backend for Django.

Requires kinterbasdb: http://kinterbasdb.sourceforge.net/
"""

from django.db.backends import util
import re

try:
    import kinterbasdb as Database
except ImportError, e:
    from django.core.exceptions import ImproperlyConfigured
    raise ImproperlyConfigured, "Error loading kinterbasdb module: %s" % e

DatabaseError = Database.DatabaseError
Database.init(type_conv=200)

try:
    # Only exists in Python 2.4+
    from threading import local
except ImportError:
    # Import copy of _thread_local.py from Python 2.4
    from django.utils._threading_local import local

class DatabaseWrapper(local):
    def __init__(self):
        self.connection = None
        self.queries = []

    def cursor(self):
        from django.conf import settings
        if self.connection is None:
            if settings.DATABASE_NAME == '':
                from django.core.exceptions import ImproperlyConfigured
                raise ImproperlyConfigured, "You need to specify DATABASE_NAME in your Django settings file."
            
            self.connection = Database.connect(
                database=settings.DATABASE_NAME,
                user=settings.DATABASE_USER,
                password=settings.DATABASE_PASSWORD,
                host=settings.DATABASE_HOST
            )
        cursor = PlaceholderStylerCursor(self.connection)
        if settings.DEBUG:
            return util.CursorDebugWrapper(cursor, self)
        return cursor

    def _commit(self):
        return self.connection.commit()

    def _rollback(self):
        if self.connection:
            return self.connection.rollback()

    def close(self):
        if self.connection is not None:
            self.connection.close()
            self.connection = None
    
    def _drop_database(self):
        return self.connection.drop_database()

class PlaceholderStylerCursor(Database.Cursor):
    """
    Django uses "format" ('%s') style placeholders, but firebird uses "qmark" ('?') style.
    This fixes it -- but note that if you want to use a literal "%s" in a query,
    you'll need to use "%%s".
    """
    def execute(self, query, params=[]):
        return Database.Cursor.execute(self, self.convert_placeholders(query, len(params)), params)

    def executemany(self, query, params=[]):
        query = self.convert_placeholders(query, len(params[0]))
        return Database.Cursor.executemany(self, query, params)

    def convert_placeholders(self, query, num_params):
        return query % tuple("?" * num_params)

supports_constraints = True

def quote_name(name):
    if name.startswith('"') and name.endswith('"'):
        return name # Quoting once is enough.
    return '"%s"' % name

def dictfetchone(cursor):
    "Returns a row from the cursor as a dict"
    return cursor.dictfetchone()

def dictfetchmany(cursor, number):
    "Returns a certain number of rows from a cursor as a dict"
    return cursor.dictfetchmany(number)

def dictfetchall(cursor):
    "Returns all rows from a cursor as a dict"
    return cursor.dictfetchall()

def get_last_insert_id(cursor, table_name, pk_name):
    cursor.execute('SELECT GEN_ID(%s, 0) FROM %s' % (quote_name("%s_%s_seq" % (table_name, pk_name)), quote_name(table_name)))
    return cursor.fetchone()[0]

def get_date_extract_sql(lookup_type, table_name):
    # lookup_type is 'year', 'month', 'day'
    return "EXTRACT(%s FROM %s)" % (lookup_type, table_name)

def get_date_trunc_sql(lookup_type, field_name):
    if lookup_type == 'year':
         sql = "EXTRACT(year FROM %s)||'-01-01 00:00:00'" % field_name
    elif lookup_type == 'month':
        sql = "EXTRACT(year FROM %s)||'-'||EXTRACT(month FROM %s)||'-01 00:00:00'" % (field_name, field_name)
    elif lookup_type == 'day':
        sql = "EXTRACT(year FROM %s)||'-'||EXTRACT(month FROM %s)||'-'||EXTRACT(day FROM %s)||' 00:00:00'" % (field_name, field_name, field_name)
    return "CAST(%s AS TIMESTAMP)" % sql

def get_limit_offset_sql(limit, offset=None):
    pass

def add_limit_offset_sql(sql, limit, offset=None):
    # this passed 'basic' test but it needs work
    return sql.replace("SELECT", "SELECT FIRST %s%s" % (limit, (offset and " SKIP %s" % offset or "")))

def get_random_function_sql():
    # First must install ib_udf (comes with the server)
    # DECLARE EXTERNAL FUNCTION rand
    #   RETURNS DOUBLE PRECISION
    #   BY VALUE ENTRY_POINT 'IB_UDF_rand' MODULE_NAME 'ib_udf';
    return "RAND()"

def get_fulltext_search_sql(field_name):
    raise NotImplementedError

def get_drop_foreignkey_sql():
    return "DROP CONSTRAINT"

OPERATOR_MAPPING = {
    'exact': '= %s',
    'iexact': 'CONTAINING %s',
    'contains': 'LIKE %s',
    'icontains': 'CONTAINING %s',
    'gt': '> %s',
    'gte': '>= %s',
    'lt': '< %s',
    'lte': '<= %s',
    'startswith': 'LIKE %s',
    'endswith': 'LIKE %s',
    'istartswith': 'LIKE %s', # hmmm
    'iendswith': 'LIKE %s', # hmmm
}
