# This dictionary maps Field objects to their associated Firebird column
# types, as strings. Column-type strings can contain format strings; they'll
# be interpolated against the values of Field.__dict__ before being output.
# If a column type is set to None, it won't be included in the output.

import sys, os.path
from kinterbasdb import connect, create_database
from django.core.management import call_command

#TODO: Implement CHECKs
DATA_TYPES = {
    'ASCIICharField':                'varchar(%(max_length)s) CHARACTER SET ASCII',
    'AutoField':                     'integer',
    'BooleanField':                  'smallint', # CHECK ("%(column)s" IN (0,1))',
    'CharField':                     'varchar(%(max_length)s)',
    'CommaSeparatedIntegerField':    'varchar(%(max_length)s) CHARACTER SET ASCII',
    'DateField':                     'date',
    'DateTimeField':                 'timestamp',
    'DecimalField':                  'numeric(%(max_digits)s, %(decimal_places)s)',
    'FileField':                     'varchar(%(max_length)s)',
    'FilePathField':                 'varchar(%(max_length)s)',
    'FloatField':                    'double precision',
    'ImageField':                    'varchar(%(max_length)s) CHARACTER SET ASCII',
    'IntegerField':                  'integer',
    'IPAddressField':                'varchar(15) CHARACTER SET ASCII',
    'NullBooleanField':              'smallint', #CHECK (("%(column)"s IN (0,1)) OR ("%(column)s" IS NULL))',
    'OneToOneField':                 'integer',
    'PhoneNumberField':              'varchar(20)', # CHARACTER SET ASCII',
    'PositiveIntegerField':          'integer', # CHECK ("%(column)s" >= 0)',
    'PositiveSmallIntegerField':     'smallint',
    'SlugField':                     'varchar(%(max_length)s)',
    'SmallIntegerField':             'smallint',
    'TextField':                     'blob sub_type text',
    'TimeField':                     'time',
    'URLField':                      'varchar(%(max_length)s) CHARACTER SET ASCII',
    'USStateField':                  'varchar(2) CHARACTER SET ASCII',
}

TEST_DATABASE_PREFIX = 'test_'

def create_test_db(settings, connection, verbosity, autoclobber):
    # KInterbasDB supports dynamic database creation and deletion 
    # via the module-level function create_database and the method Connection.drop_database.
       
    if settings.TEST_DATABASE_NAME:
        TEST_DATABASE_NAME = settings.TEST_DATABASE_NAME
    else:
        dbnametuple = os.path.split(settings.DATABASE_NAME)
        TEST_DATABASE_NAME = os.path.join(dbnametuple[0], TEST_DATABASE_PREFIX + dbnametuple[1])
    
    dsn = "localhost:%s" % TEST_DATABASE_NAME
    if settings.DATABASE_HOST:
        dsn = "%s:%s" % (settings.DATABASE_HOST, TEST_DATABASE_NAME)
    
    if os.path.isfile(TEST_DATABASE_NAME):
        sys.stderr.write("Database %s already exists\n" % TEST_DATABASE_NAME)
        if not autoclobber:
            confirm = raw_input("Type 'yes' if you would like to try deleting the test database '%s', or 'no' to cancel: " % TEST_DATABASE_NAME)
        if autoclobber or confirm == 'yes':
            if verbosity >= 1:
                print "Destroying old test database..."
            old_connection = connect(dsn=dsn, user=settings.DATABASE_USER, password=settings.DATABASE_PASSWORD)
            old_connection.drop_database()
        else:
                print "Tests cancelled."
                sys.exit(1)
            
    if verbosity >= 1:
        print "Creating test database..."
    try:
        charset = 'UNICODE_FSS'
        if hasattr(settings, 'FIREBIRD_CHARSET'):
            if settings.FIREBIRD_CHARSET == 'UTF8':
                charset='UTF8'                
        create_database("create database '%s' user '%s' password '%s' default character set %s" % \
            (dsn, settings.DATABASE_USER, settings.DATABASE_PASSWORD, charset))
    except Exception, e:
        sys.stderr.write("Got an error creating the test database: %s\n" % e)
        sys.exit(2)
            

    connection.close() 
    settings.DATABASE_NAME = TEST_DATABASE_NAME

    call_command('syncdb', verbosity=verbosity, interactive=False)

    if settings.CACHE_BACKEND.startswith('db://'):
        cache_name = settings.CACHE_BACKEND[len('db://'):]
        call_command('createcachetable', cache_name)

    # Get a cursor (even though we don't need one yet). This has
    # the side effect of initializing the test database.
    cursor = connection.cursor()

    return TEST_DATABASE_NAME

def destroy_test_db(settings, connection, old_database_name, verbosity):
    # KInterbasDB supports dynamic database deletion via the method Connection.drop_database.
    if verbosity >= 1:
        print "Destroying test database..."
    connection.drop_database()
    

