# -*- coding: utf-8 -*-
"""
VE-specific Form helpers.
"""

from django.forms import ValidationError
from django.forms.fields import RegexField, CharField, Select, EMPTY_VALUES
from django.utils.encoding import smart_unicode
from django.utils.translation import ugettext_lazy as _

class VEStateSelect(Select):
    """
    A Select widget that uses a list of Venezuelan States
    as its choices.
    """
    def __init__(self, attrs=None):
        from ve_states import STATE_CHOICES
        super(VEStateSelect, self).__init__(attrs, choices=STATE_CHOICES)

class VEPostalCodeField(RegexField):
    """
    A field that accepts a 'classic' XXXX Postal Code
    """
    default_error_messages = {
        'invalid': _("Enter a postal code in the format XXXX."),
    }

    def __init__(self, *args, **kwargs):
        super(ARPostalCodeField, self).__init__(r'^\d{4}$',
            min_length=4, max_length=4, *args, **kwargs)

    def clean(self, value):
        value = super(VEPostalCodeField, self).clean(value)
        if value in EMPTY_VALUES:
            return u''
        if len(value) not in (4, 8):
            raise ValidationError(self.error_messages['invalid'])
        if len(value) == 4:
            return u'%s' % (value)
        return value

class CIField(CharField):
    """
    A field that validates 'Cédula de Identidad' (CI) numbers.
    """
    default_error_messages = {
        'invalid': _("This field requires only numbers."),
        'max_digits': _("This field requires 7 or 8 digits."),
    }

    def __init__(self, *args, **kwargs):
        super(CIField, self).__init__(max_length=10, min_length=7, *args,
                **kwargs)

    def clean(self, value):
        """
        Value can be a string either in the [X]X.XXX.XXX or [X]XXXXXXX formats.
        """
        value = super(CIField, self).clean(value)
        if value in EMPTY_VALUES:
            return u''
        if not value.isdigit():
            value = value.replace('.', '')
        if not value.isdigit():
            raise ValidationError(self.error_messages['invalid'])
        if len(value) not in (7, 8):
            raise ValidationError(self.error_messages['max_digits'])

        return value

class RIFField(RegexField):
    """
    This field validates a RIF (Registro de Información Fiscal). A
    RIF is of the form A-XXXXXXXX-X. The last digit is a check digit.
    """
    rif_first_char = {
        'v':'1',
        'e':'2',
        'j':'3',
        'p':'4',
        'g':'5',
        }
    default_error_messages = {
        'invalid': _('Enter a valid RIF in A-XXXXXXXX-X or AXXXXXXXXX format.'),
        'checksum': _("Invalid RIF."),
    }

    def __init__(self, *args, **kwargs):
        super(RIFField, self).__init__(r'^[jJ|vV|eE]-\d{8}-\d$',*args, **kwargs)
    
    def clean(self, value):
        """Value can be either a string in A-XXXXXXXX-X or AXXXXXXXXX format"""
        value = super(RIFField, self).clean(value)
        if value in EMPTY_VALUES:
            return u''
        value, cd = self._canon(value)
        if self._calc_cd(value) != cd:
            raise ValidationError(self.error_messages['checksum'])
        return self._format(value, cd)

    def _canon(self, rif):
        rif = rif.replace('-', '')
        if rif[0].lower() in self.rif_first_char:
            rif = rif.replace(rif[0], self.rif_first_char[rif[0]])
        else:
            raise ValidationError(self.error_messages['invalid'])
        return rif[:-1], rif[-1]

    def _calc_cd(self, rif):
        mults = (4, 3, 2, 7, 6, 5, 4, 3, 2)
        tmp = sum([m * int(rif[idx]) for idx, m in enumerate(mults)])
        return str(11 - tmp % 11)

    def _format(self, rif, check_digit=None):
        if check_digit == None:
            check_digit = rif[-1]
            rif = rif[:-1]
        return u'%s%s%s' % (rif[:2], rif[2:], check_digit)